define([
    'underscore',
    'modules/form/section-view',
    'text!modules/new-appointment-request/views/common/method/templates/section-template.html',
    'text!modules/new-appointment-request/views/common/method/templates/exceeded.html',
    'text!modules/new-appointment-request/views/common/method/templates/not-available.html',
    'text!modules/new-appointment-request/views/common/method/templates/not-available-no-pact.html',
    'text!modules/new-appointment-request/views/common/method/templates/request-time-not-allowed.html',
    'text!modules/new-appointment-request/views/common/method/templates/visit-required.html',
    'json!modules/new-appointment-request/views/common/method/data/method-section.json',
], function(_,
    SectionView,
    template,
    _requestsExceededTemplate,
    _methodNotAvailableTemplate,
    _methodNotAvailableNoPactTemplate,
    _noExpressTimeTemplate,
    recentVisitRequired_,
    metadata) {
    'use strict';


    /**
     * @class CancelConfirmModel
     * @typedef {Backbone.View<Backbone.Model>}
     *
     * options: {
     *      typeOfCareName: String,
     *      isDirectSchedulingSupported: Boolean,
     *      isRequestSupported: Boolean,
     *      model: Backbone.Model,
     *      isPacNeeded: Boolean,
     *      isHistoryNeeded: Boolean,
     *      isLimitNeeded: Boolean,
     *      pacTeam: ?Backbone.Collection,
     *      patientHistory: ?Backbone.Model,
     *      requestLimit: ?Backbone.Model
     *  }
     */
    return SectionView.extend({
        CLERK_VALUE: 'clerk',
        DIRECT_VALUE: 'direct',

        templateContext: function() {
            return {legend: this.legend};
        },

        initialize: function() {
            var sectionName = 'scheduling-method-section';
            this.legend = this.getLegendFromMetadata(sectionName, metadata, false) || '';
            this.collection = this.getQuestionsFromMetadata(sectionName, metadata);
            this.initializeQuestionEventListeners();
        },


        onRender: function() {
            this._disabledButtons = {};

            // The fact that we do not have access to the any of the question views here is forcing us to fix after
            // everything has already been rendered.

            // Before rendering we already had all the information needed to fill out the templates
            // And would result in a lot less DOM manipulation if we just fill them out the way we wanted to start with
            this._checkDirectSupport();
            this._checkRequestSupport();

            this.selectDefaultSchedulingMethod();
        },

        _checkDirectSupport: function() {
            var directSelector = this._radioSelector(this.DIRECT_VALUE);
            var $directRadioButton = this.$(directSelector);
            var isDirectSupported = this.options.isDirectSchedulingSupported;

            if (isDirectSupported) {
                if (this.isClinicsEmpty(this.options.clinics)) {
                    this.setMethodNotAllowedMessage($directRadioButton, this.DIRECT_VALUE);
                } else if (this.isPacTeamMissing(this.options.institutionCode)) {
                    this.setNoPacTeamMessage($directRadioButton, this.DIRECT_VALUE);
                } else if (this.isDirectHistoryNeeded()) {
                    this.setRequiredTypeOfCareMessage($directRadioButton, this.DIRECT_VALUE);
                }
            } else {
                this.setMethodNotAllowedMessage($directRadioButton, this.DIRECT_VALUE);
            }
        },


        _checkRequestSupport: function() {
            var clerkSelector = this._radioSelector(this.CLERK_VALUE);
            var $clerkRadioButton = this.$(clerkSelector);
            var isRequestSupported = this.options.isRequestSupported;

            if (isRequestSupported) {
                if (this.model.isExpressCare()) {
                    if (this._isExpressCareAvailable($clerkRadioButton)) {
                        this._checkRequestDefaults($clerkRadioButton);
                    }
                } else {
                    this._checkRequestDefaults($clerkRadioButton);
                }
            } else {
                this.setMethodNotAllowedMessage($clerkRadioButton, this.CLERK_VALUE);
            }
        },

        _checkRequestDefaults: function($clerkRadioButton) {
            if (this.isRequestHistoryNeeded()) {
                this.setRequiredTypeOfCareMessage($clerkRadioButton, this.CLERK_VALUE);
            } else if (this.isLimitExceeded()) {
                this.setRequestLimitReachedMessage();
            }
        },

        _isExpressCareAvailable: function($clerkRadioButton) {
            var facility = this.model.facility();
            if (!facility.isExpressSupportedAtCurrentTime()) {
                this.setExpressNotAvailable($clerkRadioButton);
                return false;
            } else if (this.isPacTeamMissing(this.options.institutionCode)) {
                this.setNoPacTeamMessage($clerkRadioButton, this.CLERK_VALUE);
                return false;
            }
            return true;
        },

        isClinicsEmpty: function(clinics) {
            if (!clinics) {
                return true;
            }
            return clinics.length === 0;
        },

        /**
         * Checks if the if the information for the pact team exits at a facility
         * @param {String} facilityCode Either the 3 digit facility or 5 digit institutionCode
         * @return {boolean}
         */
        isPacTeamMissing: function(facilityCode) {
            return this.options.isPacNeeded && !this.options.pacTeam.hasPacTeamAtFacility(facilityCode);
        },

        /**
         * Checks if the patient meets the history requirement
         * @return {boolean}
         */
        isDirectHistoryNeeded: function() {
            return this.options.isDirectHistoryNeeded && !this.options.directHistory.hasRecentVisit();
        },

        /**
         * Checks if the patient meets the history requirement
         * @return {boolean}
         */
        isRequestHistoryNeeded: function() {
            return this.options.isRequestHistoryNeeded && !this.options.requestHistory.hasRecentVisit();
        },

        /**
         * Checks if the patient has exceeded the max number of requests
         * @return {boolean}
         */
        isLimitExceeded: function() {
            return this.options.isLimitNeeded && !this.options.requestLimit.hasRequestsLeft();
        },

        /**
         * This sets the "This method is only available today..." message under requests.
         * This could be optimized to occur before render if not this was not a selection view
         * @param {jquery} [$btn] The request scheduling radio button, will find on its own if not passed,
         *                        but saves execution time if a cached version can be past
         * @return {void}
         */
        setExpressNotAvailable: function($btn) {
            var noExpressTemplate = this._noExpressTimeTemplate || _.template(_noExpressTimeTemplate);
            var facility = this.model.facility();
            var expressTime = facility.get('expressTimes');
            var html = noExpressTemplate(expressTime.toJSON());

            var $button = this._ensureExists($btn, this.CLERK_VALUE);
            var $parent = $button.parent();

            this._disabledButtons[this.CLERK_VALUE] = true;
            this._disableButton($button);
            this._noExpressTimeTemplate = noExpressTemplate;

            $parent.append(html);
        },

        /**
         * This sets the no pact team message under the direct scheduling button.
         * This could be optimized to occur before render if not this was not a selection view
         * @param {jquery} [$btn] The direct scheduling radio button, will find on its own if not passed,
         *                        but saves execution time if a cached version can be past
         * @param {!string} method this.DIRECT_VALUE or this.CLERK_VALUE
         * @return {void}
         */
        setNoPacTeamMessage: function($btn, method) {
            var html = this.getNoPactHtml(method);

            var $button = this._ensureExists($btn, method);
            var $parent = $button.parent();

            this._disabledButtons[method] = true;
            this._disableButton($button);
            $parent.append(html);
        },

        /**
         * Sets the method not allowed message under one of the radio buttons
         * This could be optimized to occur before render if not this was not a selection view
         *
         * @example setRequestLimitReachedMessage(button, 'clerk')
         * @example setRequestLimitReachedMessage(null, this.CLERK_VALUE)
         *
         * @param {?jquery} [$btn] The Clerk or Direct Radio Button
         * @param {!string} method this.DIRECT_VALUE or this.CLERK_VALUE
         * @return {void}
         */
        setMethodNotAllowedMessage: function($btn, method) {
            var html = this.getNotAllowedHtml();
            var $button = this._ensureExists($btn, method);
            var $parent = $button.parent();

            this._disabledButtons[method] = true;
            this._disableButton($button);
            $parent.append(html);
        },

        /**
         * Sets the request limited exceeded message under the clerk button
         * This could be optimized to occur before render if not this was not a selection view
         * @param {?jquery} [$btn] The request scheduling `clerk` button will find on its own if not passed,
         *                         but saves execution time if a cached version can be past
         * @return {void}
         */
        setRequestLimitReachedMessage: function($btn) {
            var $button = this._ensureExists($btn, this.CLERK_VALUE);
            var requestsExceededTemplate = _.template(_requestsExceededTemplate);
            var requests = this.options.requestLimit;
            var typeOfCare = this.options.typeOfCareName;
            var data = {
                typeOfCareName: typeOfCare,
                submittedRequestLimit: requests.get('requestLimit'),
            };
            var html;

            _.extend(data, this.templateContext());
            html = requestsExceededTemplate(data);

            this._disabledButtons[this.CLERK_VALUE] = true;
            this._disableButton($button, html);
        },

        /**
         * Sets the message saying that you must have received this type of care in the last 12/24 months
         * This could be optimized to occur before render if not this was not a selection view
         *
         * @example setRequestLimitReachedMessage(button, 'clerk')
         * @example setRequestLimitReachedMessage(null, this.CLERK_VALUE)
         *
         * @param {?jquery} [$btn] The Clerk or Direct Radio Button
         * @param {!string} method this.DIRECT_VALUE or this.CLERK_VALUE
         * @return {void}
         */
        setRequiredTypeOfCareMessage: function($btn, method) {
            var $button = this._ensureExists($btn, method);
            var history = method === this.DIRECT_VALUE ? this.options.directHistory : this.options.requestHistory;
            var duration = history.get('durationInMonths');
            var html = this.getRequireTypeOfCareHtml(duration);
            var $parent = $button.parent();

            this._disabledButtons[method] = true;
            this._disableButton($button);
            $parent.append(html);
        },

        /**
         * Ensures that the "no pact team" template is only complied once, and builds the HTML with it.
         * @param {string} schedulingMethod
         * @return {string} HTML
         */
        getNoPactHtml: function(schedulingMethod) {
            this._templateNoPact = this._templateNoPact || _.template(_methodNotAvailableNoPactTemplate);
            return this._templateNoPact({schedulingMethod: schedulingMethod});
        },

        /**
         * Ensures that the "method not allowed" template is only complied once, and builds the HTML with it.
         * @return {string} HTML
         */
        getNotAllowedHtml: function() {
            this._templateNotAvailable = this._templateNotAvailable || _.template(_methodNotAvailableTemplate);
            return this._templateNotAvailable();
        },

        /**
         * @param {number} numberOfMonths The limit to the amount of time between visits for the selected type of care
         * @return {*}
         */
        getRequireTypeOfCareHtml: function(numberOfMonths) {
            this._templateRequireTypeOfCare = this._templateRequireTypeOfCare || _.template(recentVisitRequired_);
            return this._templateRequireTypeOfCare({numberOfMonths: numberOfMonths});
        },

        /**
         * @param {jQuery} [$directButton]
         * @param {jQuery} [$clerkButton]
         * @return {void}
         */
        selectDefaultSchedulingMethod: function($directButton, $clerkButton) {
            var $button;

            if (this._disabledButtons[this.DIRECT_VALUE] && !this._disabledButtons[this.CLERK_VALUE]) {
                $button = this._ensureExists($clerkButton, this.CLERK_VALUE);
                $button.prop('checked', true);
                this.model.set('scheduling-method', this.CLERK_VALUE);
            } else if (!this._disabledButtons[this.DIRECT_VALUE] && this._disabledButtons[this.CLERK_VALUE]) {
                $button = this._ensureExists($directButton, this.DIRECT_VALUE);
                $button.prop('checked', true);
                this.model.set('scheduling-method', this.DIRECT_VALUE);
            }
        },

        /**
         * @param {jQuery} $button
         * @param {string} html
         * @return {void}
         * @private
         */
        _disableButton: function($button, html) {
            var $label = $button.parent('label');

            $button.prop('disabled', true);
            $label.addClass('ui-disabled');
            if (html) {
                $label.append(html);
            }
        },

        /**
         * Check that the button has a value and finds it does not/
         * This was added to ensure that the changes too the file are reverse compatible with past implementations
         *
         * @param {jQuery} $button
         * @param {string} valueName The buttons `value` field
         * @return {*}
         * @private
         */
        _ensureExists: function($button, valueName) {
            var selector;

            if ($button && $button.length) {
                return $button;
            }
            selector = this._radioSelector(valueName);
            return this.$el.find(selector);
        },

        /**
         * Helper function to build the jQuery selector
         * @param {string} value The buttons `value` attribute
         * @return {string}
         * @private
         */
        _radioSelector: function(value) {
            return 'input[type="radio"][value="' + value + '"]';
        },
    });
});
